/*
  test-store is a program to manipulate a store from the command line.
  Its purpose is mainly to test the implementation in s-store.c.

  Usage: test-store name1 name2 [command [arg ...] [command [arg ...] ...]

  The name1 and name2 represent the local and remote targets that
  identify a store. The other arguments are zero or more commands with
  their arguments. The commands are:

  get PATH -- retrieves the entry for PATH and prints it. Output is
  either a line "MODE TIME SIZE" (with MODE an octal number
  representing the file's mode, TIME a decimal number representing a
  time stamp, and SIZE a decimal file size). or a line "Not found"

  put PATH MODE TIME SIZE -- puts a new entry for PATH in the store,
  or updates an existing entry. MODE must be an octal number, TIME and
  SIZE must be decimal numbers.

  del PATH -- deletes the entry for PATH from the store.

  list -- list all entries in the store, each entry resulting in a
  line "MODE TIME SIZE PATH".

  reset -- make the store empty (delete all entries).

  Copyright © 2018 Bert Bos <bert@phonk.net>
  Created: 18 February 2018
  Author: Bert Bos <bert@phonk.net>
*/

#include "stdincls.h"
#include "types.e"
#include "print.e"
#include "s-store.e"


static void usage(const char *progname)
{
  fprintf(stderr,
	  "Usage: %s name1 name2 [command [arg ...] [command [arg ...] ...]\n",
	  progname);
  exit(1);
}


int main(int argc, char *argv[])
{
  DB_cursor cursor;
  fileinfo info;
  bool created;
  DB db;
  int i;

  if (argc < 2) usage(argv[0]);
  db = store_open(argv[1], argv[2], &created);
  printf("%s\n", created ? "Store created" : "Store opened");

  for (i = 3; i < argc; i++) {

    if (strcmp(argv[i], "get") == 0) {
      if (++i >= argc) errx(EX_USAGE, "Missing PATH argument after `get'");
      if (!store_get(db, argv[i], &info))
	printf("Not found\n");
      else
	printf("%o %ld %lld\n", info.mode, info.time, info.size);

    } else if (strcmp(argv[i], "put") == 0) {
      if (++i >= argc) errx(EX_USAGE, "Missing PATH argument after `put'");
      info.path = strdup(argv[i]);
      if (++i >= argc) errx(EX_USAGE, "Missing MODE argument after `put'");
      info.mode = strtoul(argv[i], NULL, 8);
      if (++i >= argc) errx(EX_USAGE, "Missing TIME argument after `put'");
      info.time = strtol(argv[i], NULL, 10);
      if (++i >= argc) errx(EX_USAGE, "Missing SIZE argument after `put'");
      info.size = strtoll(argv[i], NULL, 10);
      info.status = '-';	/* Indicate that info.sums is not used */
      if (!store_put(db, info)) err(EX_OSERR, NULL);

    } else if (strcmp(argv[i], "del") == 0) {
      if (++i >= argc) errx(EX_USAGE, "Missing PATH argument after `del'");
      (void) store_del(db, argv[i]);

    } else if (strcmp(argv[i], "list") == 0) {
      if (!(cursor = cursor_create(db))) err(EX_OSERR, NULL);
      while (cursor_next(cursor, &info))
	printf("%o %ld %lld %s\n", info.mode, info.time, info.size, info.path);

    } else if (strcmp(argv[i], "reset") == 0) {
      store_reset(db);

    } else {
      err(EX_USAGE, "Unknown command `%s'", argv[i]);
    }
  }

  if (!store_close(db)) err(EX_OSERR, NULL);
  return 0;
}
