/*
  make_symlink() is called by serve1() to make or update a
  symlink. The target of the symlink is read from stream in.

  The path is deleted and recreated as a symlink. However, if the
  existing path represents a directory, it is not replaced by a
  symlink, but an error is returned.

  Implementation:

  make_symlink() sends "OK" to the client in case of success, or an
  error line in case of failire.
*/

#include "stdincls.h"
#include "errcodes.e"
#include "types.e"
#include "getline.e"
#include "print.e"
#include "s-store.e"
#include "s-updatelog.e"


/* make_symlink -- make path into a symlink */
EXPORT void make_symlink(FILE *in, FILE *out, const char *root,
			 const time_t modtime, char *file, const DB store)
{
  char path[FILENAME_MAX+1];
  struct timeval times[2];
  struct stat buf;
  fileinfo info;
  char *line;

  if (!make_absolute(root, file, path)) {
    print(out, "? %03d Path too long\n", EC_TOOLONG);
    return;
  }

  /* Create the symlink. (glibc < 2.6 doesn't implement lutimes() and
     returns errno==ENOSYS. It means the time stamps will not be
     synchronized, but that is not a big problem.)*/
  times[0].tv_sec = time(NULL); times[0].tv_usec = 0;
  times[1].tv_sec = modtime; times[1].tv_usec = 0;
  if ((line = getline_chomp(in)) &&			   /* Read target */
      create_directories(path) &&			   /* Create dirs */
      (unlink(path) == 0 || errno == ENOENT) &&		   /* Remove old file */
      symlink(line, path) == 0 &&			   /* Create symlink */
      (lutimes(path, times) == 0 || errno == ENOSYS) &&	   /* Set mod. time */
      lstat(path, &buf) == 0) {				   /* Get mode bits */

    info.status = '-';		/* Indicate that info.sums is not used */
    info.mode = buf.st_mode;
    info.time = buf.st_mtime;
    info.size = buf.st_size;
    info.path = file;
    if (store_put(store, info)) {
      print1(out, "OK\n");
      return;
    }
  }
  print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
}
