/*
  init_profile() is called by main() to set the fields of a profile to
  default values or values indicating that they are not yet set.

  read_profile() is called by main() to read a stored profile with the
  roots to synchronize and the file patterns to skip.

  add_exclusion() is called by main() and by determine_file_actions()
  to add a pattern to a profile for paths that should be skipped. If
  the profile has a name, the exclusion is also added to the profile
  on disk.

  add_inclusion() is called by main() to add a pattern to a profile
  for file paths that should not be skipped, despite matching an
  exclusion pattern. If the profile has a name, the exclusion is also
  added to the prfile on disk.

  Command line options (-e and -i) are processed before any profile
  name is parsed and thus command line options are not stored in the
  profile on disk. On the other hand, user commands during the
  interactive determination of file actions result in patterns being
  added to the stored profile, if the profile has a name.

  is_excluded() is called by determine_file_actions() to check if a
  file should be skipped.

  The order of exclusions and inclusions is thus important. The last
  pattern that matches determines what to do. If it is an exclusion
  pattern, the file is skipped. If it is an inclusion patterns, the
  file is processed. If no pattern matches, the file is processed.

  Exclusion and inclusion patterns from add_exclusion() and
  add_inclusion() override (come after) patterns read from a stored
  profile.

  Note that there is currently no way to add inclusion patterns to a
  stored profile (they can only be given on the command line) and
  there is also no way to remove or edit patterns once added. But the
  profile can be edited with a text editor.

  The profile struct contains a linked list of patterns in reverse
  order, i.e., the last patterns are at the start of the list.
*/

#include "stdincls.h"
#include "types.e"
#include "getline.e"
#include "print.e"
#include "homedir.e"

EXPORT typedef struct pattern {
  enum {INCLUDE, EXCLUDE} type;
  char *pattern;		/* Shell pattern, see fnmatch(3) */
  struct pattern *next;
} *Pattern;

EXPORT typedef struct profile {
  char *profile;		/* Name relative to ~/r2sync/ or NULL */
  char *root1, *root2;		/* Targets to sync, may include host names */
  char id1[MACHINE_ID_LEN+1];	/* Globally unique ID of root1 */
  char id2[MACHINE_ID_LEN+1];	/* Globally unique ID of root2 */
  char *path1, *path2;		/* Absolute paths on root1, resp. root2 */
  filetype type1, type2;	/* Type (directory or file) of path1 & path2 */
  pid_t child1, child2;		/* Process IDs of the two servers */
  FILE *to1, *from1;		/* Pipes to and from server 1 */
  FILE *to2, *from2;		/* Pipes to and from server 2 */
  int protocol;			/* Protocol version to use with servers */
  bool batch_mode, quiet;	/* Command line options -b and -q */
  bool reset_logs, statistics;	/* Options -r and -s */
  bool compression, keep_going;	/* Options -C and -k */
  bool use_shortcuts;		/* Negated by option -n */
  bool nocase;			/* True if a target is case-insensitive */
  Pattern patterns;		/* Patterns to include/eclude files by name */
  time_t start;			/* For progress meter: start time of sync */
  long long ntotal, ntodo;	/* For progress meter: total & remaining bytes */
  long long sent1, sent2;	/* For statistics: initial bytes sent */
  char conflict_resolution;	/* Option -c */
} *Profile;


/* profile_path -- construct the full path of a profile, allocated on heap */
static char *profile_path(const char *profile)
{
  char *home, *s;
  size_t n;

  if (!(home = home_dir())) return NULL;
  n = strlen(home) + sizeof(R2SYNCDIR) + strlen(profile) + 7;
  if (!(s = malloc(n * sizeof(*s)))) return NULL;
  sprintf(s, "%s/%s/%s.prf", home, R2SYNCDIR, profile);
  return s;
}


/* init_profile -- intialize the fields of a profile */
EXPORT void init_profile(Profile p)
{
  p->profile = p->root1 = p->root2 = p->path1 = p->path2 = NULL;
  p->id1[0] = p->id2[0] = '\0';
  p->type1 = p->type2 = IS_OTHER;
  p->child1 = p->child2 = 0;
  p->to1 = p->from1 = p->to2 = p->from2 = NULL;
  p->protocol = R2SYNCPROTO;
  p->batch_mode = p->quiet = p->reset_logs = p->statistics = false;
  p->compression = p->nocase = p->keep_going = false;
  p->use_shortcuts = true;
  p->patterns = NULL;
  p->start = 0;
  p->ntotal = p->ntodo = 0;
  p->conflict_resolution = '/';
}


/* add_exclusion -- add an exclusion pattern to the profile */
EXPORT void add_exclusion(Profile profile, const char *pattern)
{
  Pattern h;

  if (!(h = malloc(sizeof(*h))) || !(h->pattern = strdup(pattern)))
    err(EX_OSERR, NULL);
  h->type = EXCLUDE;
  h->next = profile->patterns;
  profile->patterns = h;
}


/* add_inclusion -- add an inclusion pattern to the profile */
EXPORT void add_inclusion(Profile profile, const char *pattern)
{
  Pattern h;

  if (!(h = malloc(sizeof(*h))) || !(h->pattern = strdup(pattern)))
    err(EX_OSERR, NULL);
  h->type = INCLUDE;
  h->next = profile->patterns;
  profile->patterns = h;
}


/* set_bool -- parse a boolean value */
static bool set_bool(const char *key, const char *value)
{
  if (strchr("yYtT1", *value))
    return true;
  else if (strchr("nNfF0", *key))
    return false;
  else
    errx(EX_CONFIG,
	 _("Error in profile: %1$s should be yes/no/true/false/1/0, found `%2$s'"),
	 key, value);
}


/* read_profile -- read roots and exceptions from a profile file */
EXPORT void read_profile(Profile profile)
{
  char *s = NULL, *line, *key, *value;
  FILE *f;

  assert(profile->profile);

  /* Construct path name and open the profile file */
  s = profile_path(profile->profile);
  if (!(f = fopen(s, "r"))) err(EX_IOERR, "%s", s);

  /* Parse the lines in the profile file */
  profile->root1 = profile->root2 = NULL;
  while ((line = getline_chomp(f))) {
    key = line + strspn(line, " \t");
    value = key + strcspn(key, " \t=");
    value += strspn(value, " \t=");
    if (*key == '\0' || *key == '#') {
      /* Empty line or comment, skip */
    } else if (strncmp(key, "root", 4) == 0) {
      if (!profile->root1) {
	profile->root1 = strdup(value);
	if (!profile->root1) err(EX_OSERR, NULL);
      } else if (!profile->root2) {
	profile->root2 = strdup(value);
	if (!profile->root2) err(EX_OSERR, NULL);
      } else
	errx(EX_CONFIG, _("Too many `root' lines in profile file %s"), s);
    } else if (strncmp(key, "exclude", 7) == 0) {
      add_exclusion(profile, value);
    } else if (strncmp(key, "include", 7) == 0) {
      add_inclusion(profile, value);
    } else if (strncmp(key, "compression", 11) == 0) {
      profile->compression = set_bool("compression", value);
    } else if (strncmp(key, "batch", 5) == 0) {
      profile->batch_mode = set_bool("batch", value);
    } else if (strncmp(key, "quiet", 5) == 0) {
      profile->quiet = set_bool("quiet", value);
    } else if (strncmp(key, "statistics", 10) == 0) {
      profile->statistics = set_bool("statistics", value);
    } else if (strncmp(key, "conflict-resolution", 19) == 0) {
      profile->conflict_resolution = *value;
    } else if (strncmp(key, "protocol", 8) == 0) {
      profile->protocol = atoi(value);
    } else {
      errx(EX_CONFIG, _("Incorrect line in profile %1$s: %3$.*2$s\n"),
	   s, (int)(value - key), key);
    }
    /* The syntax of values not checked here will be checked in main() */
  }
  if (ferror(f)) err(EX_IOERR, "%s", s);
  fclose(f);

  if (!profile->root2)
    errx(EX_CONFIG, _("There must be two `root' lines in profile file %s"), s);

  free(s);
}


/* is_excluded -- check if a path is excluded by the patterns in a profile */
EXPORT bool is_excluded(Profile profile, const char *path)
{
  Pattern h = profile->patterns;

  while (h && fnmatch(h->pattern, path, 0) != 0) h = h->next;
  return h && h->type == EXCLUDE ? true : false;
}
